using System;
using System.Collections;
using System.Windows.Forms;
using System.Drawing;
using CsGL.OpenGL;

namespace SPStudio
{
	/// <summary>
	/// Handles the display of Renderable Objects.
	/// </summary>
	public class OpenGLPanel : OpenGLControl
	{
		#region Public Vars
        public ArrayList renderables;
		public Size size;
		public double far;
		public double FOV;
		#endregion

		#region Private Vars
		/// <summary>
		/// Used to determine how
		/// to display the Renderables.
		/// </summary>
		private bool[] state = new bool[8]{false,false,false,false,true,true,true,true};

		/// <summary>
		/// View orientation variables.
		/// </summary>
		private double rotateY = 0.0;
		private double rotateX = 0.0;
		private float newX = 0.0f;
		private float newY = 0.0f;
		#endregion

		#region Constructor
		/// <summary>
		/// Obtain handle to an ArrayList of Renderables.
		/// Gather needed information from the Parent
		/// workspace.
		/// </summary>
		/// <param name="Renderables"></param>
		/// <param name="Parent"></param>
		public OpenGLPanel(ArrayList Renderables, Workspace Parent): base()
		{
			far = Parent.distance_to_camera*2.0;
			size = Parent.getSize();
			renderables = Renderables;
			FOV = Parent.cameraFOV;
			this.KeyDown += new KeyEventHandler(OpenGLPanel_KeyDown);
		}
		#endregion

		#region Main Draw method
		/// <summary>
		/// Handle the drawing of the Renderables
		/// </summary>
		public override void glDraw()
		{     
			GL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT); // Clear The Screen And The Depth Buffer  
			GL.glMatrixMode(GL.GL_MODELVIEW); // Select The Modelview Matrix
			GL.glLoadIdentity();                 // Reset The Current Modelview Matrix         
			
			GL.gluLookAt(0.0,0.0,(far/2.0),0.0,0.0,0.0,0.0,1.0,0.0);
			GL.glTranslatef(newX, newY, 0.0f);
			GL.glRotatef((float)rotateY,0.0f,1.0f,0.0f);
			GL.glRotatef((float)rotateX,1.0f,0.0f,0.0f);

			for(int index=0; index<renderables.Count; index++)
			{
				((Renderable)renderables[index]).OpenGLDraw(state);
				GL.glFlush();
			}
		}
		#endregion

		#region Methods to Initialize the OpenGL environment
		/// <summary>
		/// Initialize the OpenGL environment
		/// </summary>
		protected override void InitGLContext() 
		{
			GL.glShadeModel(GL.GL_SMOOTH);         // Set Smooth Shading                 
			GL.glClearColor(0.0f, 0.0f, 0.0f, 0.5f);     // BackGround Color         
			GL.glClearDepth(1.0f);                 // Depth buffer setup    
			GL.glEnable(GL.GL_BLEND);
			GL.glBlendFunc(GL.GL_SRC_ALPHA,GL.GL_ONE);
			GL.glEnable(GL.GL_DEPTH_TEST);         // Enables Depth Testing 
			GL.glDepthFunc(GL.GL_LEQUAL);             // The Type Of Depth Test To Do     
			GL.glHint(GL.GL_PERSPECTIVE_CORRECTION_HINT, GL.GL_NICEST);     /* Really Nice Perspective Calculations */     
		}

		/// <summary>
		/// Configure the OpenGL environment
		/// in order to properly display
		/// the Renderable Objects by
		/// using information obtained from
		/// the parent workspace.
		/// </summary>
		/// <param name="e"></param>
		protected override void OnSizeChanged(EventArgs e)
		{
			base.OnSizeChanged(e);
        
			double aspect_ratio;

			aspect_ratio =  (double) size.Width / (double) size.Height;

			GL.glMatrixMode(GL.GL_PROJECTION); // Select The Projection Matrix
			GL.glLoadIdentity(); // Reset The Projection Matrix
            
			// Calculate The Aspect Ratio Of The Window
			GL.gluPerspective(FOV, aspect_ratio, 0.1f, far);
			//GL.gluPerspective(FOV, aspect_ratio, far-10.0, far);
    
			GL.glMatrixMode(GL.GL_MODELVIEW); // Select The Modelview Matrix
			GL.glLoadIdentity();// Reset The Modelview Matrix
		}
		#endregion

		#region Keyboard Controls
		/// <summary>
		/// Read in key presses and 
		/// orient the view accordingly.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OpenGLPanel_KeyDown(object sender, KeyEventArgs e)
		{
			//A,S,W,Z rotate the camera
			if (e.KeyCode == Keys.A) 
			{
				rotateY-=10.0;
				if (rotateY>=360.0) rotateY = 0.0;
				if (rotateY<=-360.0) rotateY = 360.0;
			}
			if (e.KeyCode == Keys.S) 
			{
				rotateY+=10.0;
				if (rotateY>=360.0) rotateY = 0.0;
				if (rotateY<=-360.0) rotateY = 360.0;
			}
			if (e.KeyCode == Keys.W) 
			{
				rotateX-=10.0;
				if (rotateX>=360.0) rotateX = 0.0;
				if (rotateX<=-360.0) rotateX = 360.0;
			}
			if (e.KeyCode == Keys.Z) 
			{
				rotateX+=10.0;
				if (rotateX>=360.0) rotateX = 0.0;
				if (rotateX<=-360.0) rotateX = 360.0;
			}
			//I,M,J,K translate the camera
			if (e.KeyCode == Keys.I) newY+=1.0f;
			if (e.KeyCode == Keys.M) newY-=1.0f;
			if (e.KeyCode == Keys.J) newX-=1.0f;
			if (e.KeyCode == Keys.K) newX+=1.0f;
			Refresh();
		}
		#endregion

		#region Update State
		/// <summary>
		/// Replace the current boolean state array
		/// with boolean array sent in.
		/// The input param is expected to
		/// be a bool[8].
		/// 1-4 indicate isolation displays.
		/// 5-8 indicate intersection displays.
		/// </summary>
		/// <param name="State"></param>
		public void updateState(bool[] State)
		{
			//State is expected to be a bool[8] array
			state = State;
			Refresh();
		}
		#endregion
	}
}
